-- Tabelle für DB-Logmessages
    -- DROP TABLE IF EXISTS TSystem.Log;
CREATE TABLE TSystem.Log (
  log_id         SERIAL PRIMARY KEY,                                --
  log_tx_id      BIGINT DEFAULT txid_current(),                     -- Transaktions-ID der Transaktion, die den Logeintrag ausgelöst hat.
  log_timestamp  TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),   -- Zeitstempel
  log_user       VARCHAR(30) DEFAULT tsystem.current_user_ll_db_usename(true),         -- DB-Benutzer dessen Session aktiv war. (Datentyp NAME statt Varchar, spart Typecast)
  log_level      INTEGER,                                           -- Loglevel, analog Delphi. Als Integer codiert.
  log_category   INTEGER,                                           -- Zuordnung zu einer bestimmten Gruppe. Z.Bsp. Debug-Nachricht oder EAV, derzeit nicht spezifiziert
  log_source     VARCHAR(250),                                      -- Ursprung des Log-Eintrags. Standard ist Name der aufrufenden Funktion.
  log_context    TEXT,                                              -- Aufrufkontext, wo in der aufrufenden Funktion oder z.Bsp. Eingangsparameter
  log_message    TEXT                                               -- Die eigentliche Lognachricht
);

-- Indizes
    CREATE INDEX log_user ON TSystem.Log(log_user);
--

-- Initialisierung von DB-Debuglog und Tabellenlog. Von Delphi im OnConnect aufgerufen.
CREATE OR REPLACE FUNCTION TSystem.Log_Init() RETURNS VOID AS $$
  DECLARE level INTEGER;
          connstring VARCHAR;
  BEGIN
    -- Session-Variable: Auditlog-Deaktivierungszähler. Von log_audit_disable hochgezählt von log_audit_enable runtergezählt. Tabellen-Logeinträge werden nur geschrieben, wenn das 0 ist.
    PERFORM set_config('prodat.log_audit_count', '0', false);

    -- Asynchrones Loggen über DB-Link (de)aktivieren, ASynchron erhält Logeinträge beim Rollback auf Nutzer-Session.
    PERFORM set_config('prodat.log_async', TSystem.Settings__GetBool('prodat.log_async', false)::VARCHAR, false);

    -- DBLink-Verbindung aufbauen
    IF current_setting('prodat.log_async') THEN
        connstring:= 'host=localhost port=' || inet_server_port() || ' dbname = ' || current_catalog || ' user=' || current_user || ' password= application_name=' || quote_literal('PRODAT-ERP.DBLink-Log');
        BEGIN
            PERFORM dblink_connect('dblink_logmsg', connstring);  -- DB-Link für Log-Message-Tabelle
        EXCEPTION WHEN OTHERS THEN
            PERFORM dblink_disconnect('dblink_logmsg');
            PERFORM dblink_connect('dblink_logmsg', connstring);  -- DB-Link für Log-Message-Tabelle
        END;
    END IF;

    -- Loglevel der Session setzen. Nach Nutzereinstellung, wenn nicht vorhanden dann Systemdefault.
    level := Log_Get_LogLevel(current_user::VARCHAR);

    PERFORM set_config('prodat.sessions_loglevel', level::VARCHAR, false);

    RETURN;
  END $$ LANGUAGE plpgsql;
--

-- Basisfunktion um Logeintrag zu erstellen. Alle Ableitungen gehen hierauf zurück.
CREATE OR REPLACE FUNCTION TSystem.LogMessage(loglevel INTEGER, source VARCHAR(100), message TEXT, category INTEGER DEFAULT 0, context TEXT DEFAULT NULL) RETURNS VOID AS $$
  DECLARE level INTEGER;
        curr_loglevel INTEGER;
        sql TEXT;
  BEGIN
    BEGIN
        curr_loglevel:= current_setting('prodat.sessions_loglevel')::INTEGER;
    EXCEPTION WHEN OTHERS THEN
        -- TODO: Weg finden, das ohne Exceptionblog aufzurufen. Die sind normalerweise relativ langsam. (LG, 06/2017)
        --       DS PostgreSQL 9.6 > current_setting(setting_name [, missing_ok ])
        PERFORM TSystem.Log_Init();
        curr_loglevel:=current_setting('prodat.sessions_loglevel')::INTEGER;
    END;

    -- Loglevel prüfen ob das wichtig genug ist und geschrieben werden soll.
    IF loglevel <= curr_loglevel OR loglevel = 6 THEN -- Level 6 ist "Always" ... das wird immer geschrieben.
        IF current_setting('prodat.log_async') THEN
            -- Insert Statement zusammenbauen und asynchron loggen.
            sql:= FORMAT('INSERT INTO TSystem.Log(log_level, log_category, log_context, log_message, log_source, log_tx_id) VALUES(%L, %L, %L, %L, %L, %L)', loglevel, 0, context, message, source, txid_current());
            PERFORM * FROM dblink('dblink_logmsg', sql) AS t1(test text);
        ELSE
            INSERT INTO TSystem.Log(log_level, log_category, log_context, log_message, log_source) VALUES(loglevel, 0, context, message, source);
        END IF;
    END IF;

    RETURN;
  END $$ LANGUAGE plpgsql;
--

-- Aktuelles Loglevel für einen Benutzer ermitteln ( Nutzerlevel abfragen, wenn nicht gesetzt auf System-Loglevel zurückgehen)
CREATE OR REPLACE FUNCTION TSystem.Log_Get_LogLevel(IN _user VARCHAR DEFAULT '') RETURNS INTEGER AS $$
  DECLARE currLevel INTEGER;
           my_s_vari VARCHAR(70);
  BEGIN
    IF _user = '' THEN
        my_s_vari:= 'SysLogLevel';
    ELSE
        IF EXISTS(SELECT TRUE FROM settings WHERE s_vari = 'UserLogLevel_' || TRIM(UPPER(current_user::VARCHAR))) THEN
            my_s_vari:= 'UserLogLevel_' || TRIM(UPPER(current_user::VARCHAR));
        ELSE
            my_s_vari:= 'SysLogLevel';
        END IF;
    END IF;

    CASE TSystem.Settings__Get(my_s_vari)
        WHEN 'pllFatalError'    THEN currLevel:= 0;
        WHEN 'pllError'         THEN currLevel:= 1;
        WHEN 'pllWarning'       THEN currLevel:= 2;
        WHEN 'pllInfo'          THEN currLevel:= 3;
        WHEN 'pllDebug'         THEN currLevel:= 4;
        WHEN 'pllDebugVerbose'  THEN currLevel:= 5;
        WHEN 'pllAlways'        THEN currLevel:= 6;
    ELSE
        currLevel:= 0;
    END CASE;

    RETURN currLevel;
  END $$ LANGUAGE plpgsql STABLE;
--

-- Loglevel für einen Benutzer setzen
CREATE OR REPLACE FUNCTION TSystem.Log_Set_LogLevel(IN _user VARCHAR DEFAULT '', currLevel INTEGER DEFAULT 0) RETURNS VOID AS $$
  DECLARE my_s_vari VARCHAR(70);
          my_s_inha VARCHAR;
  BEGIN
    IF _user = '' THEN
        my_s_vari:= 'SysLogLevel';
    ELSE
        my_s_vari:= 'UserLogLevel_' || TRIM(UPPER(current_user::VARCHAR));
    END IF;

    CASE currLevel
        WHEN 0 THEN my_s_inha:= 'pllFatalError';
        WHEN 1 THEN my_s_inha:= 'pllError';
        WHEN 2 THEN my_s_inha:= 'pllWarning';
        WHEN 3 THEN my_s_inha:= 'pllInfo';
        WHEN 4 THEN my_s_inha:= 'pllDebug';
        WHEN 5 THEN my_s_inha:= 'pllDebugVerbose';
        WHEN 6 THEN my_s_inha:= 'pllAlways';
    END CASE;

    PERFORM TSystem.Settings__Set(my_s_vari, my_s_inha);
    PERFORM set_config('prodat.sessions_loglevel', currLevel, false);

    RETURN;
  END $$ LANGUAGE plpgsql;
--

-- Überladene Logfunktionen nach Loglevel
    --
    CREATE OR REPLACE FUNCTION TSystem.LogFatalError(source VARCHAR, message TEXT, category INTEGER DEFAULT 0, context TEXT DEFAULT NULL) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(0, source, message, category, context);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    --
    CREATE OR REPLACE FUNCTION TSystem.LogError(source VARCHAR, message TEXT, category INTEGER DEFAULT 0, context TEXT DEFAULT NULL) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(1, source, message, category, context);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    --
    CREATE OR REPLACE FUNCTION TSystem.LogWarning(source VARCHAR, message TEXT, category INTEGER DEFAULT 0, context TEXT DEFAULT NULL) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(2, source, message, category, context);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    --
    CREATE OR REPLACE FUNCTION TSystem.LogInfo(source VARCHAR, message TEXT, category INTEGER DEFAULT 0, context TEXT DEFAULT NULL) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(3, source, message, category, context);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    --
    CREATE OR REPLACE FUNCTION TSystem.LogDebug(source VARCHAR, message TEXT, category INTEGER DEFAULT 0, context TEXT DEFAULT NULL) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(4, source, message, category, context);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    --
    CREATE OR REPLACE FUNCTION TSystem.LogDebugVerbose(source VARCHAR, message TEXT, category INTEGER DEFAULT 0, context TEXT DEFAULT NULL) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(5, source, message, category, context);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    --
    CREATE OR REPLACE FUNCTION TSystem.LogAlways(source VARCHAR, message TEXT, category INTEGER DEFAULT 0, context TEXT DEFAULT NULL) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(6, source, message, category, context);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    -- Log-Funktionen Message-Texte als VARIADIC und Textformatierung TSystem.LogFormat
    CREATE OR REPLACE FUNCTION TSystem.LogFatalError(source VARCHAR, message TEXT, VARIADIC arr_msg_txt VARCHAR[]) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(0, source, TSystem.LogFormat(message, False, VARIADIC arr_msg_txt), 0, NULL);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    --
    CREATE OR REPLACE FUNCTION TSystem.LogError(source VARCHAR, message TEXT, VARIADIC arr_msg_txt VARCHAR[]) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(1, source, TSystem.LogFormat(message, False, VARIADIC arr_msg_txt), 0, NULL);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    --
    CREATE OR REPLACE FUNCTION TSystem.LogWarning(source VARCHAR, message TEXT, VARIADIC arr_msg_txt VARCHAR[]) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(2, source, TSystem.LogFormat(message, False, VARIADIC arr_msg_txt), 0, NULL);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    --
    CREATE OR REPLACE FUNCTION TSystem.LogInfo(source VARCHAR, message TEXT, VARIADIC arr_msg_txt VARCHAR[]) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(3, source, TSystem.LogFormat(message, False, VARIADIC arr_msg_txt), 0, NULL);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    --
    CREATE OR REPLACE FUNCTION TSystem.LogDebug(source VARCHAR, message TEXT, VARIADIC arr_msg_txt VARCHAR[]) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(4, source, TSystem.LogFormat(message, False, VARIADIC arr_msg_txt), 0, NULL);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    --
    CREATE OR REPLACE FUNCTION TSystem.LogDebugVerbose(source VARCHAR, message TEXT, VARIADIC arr_msg_txt VARCHAR[]) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(5, source, TSystem.LogFormat(message, False, VARIADIC arr_msg_txt), 0, NULL);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --

    --
    CREATE OR REPLACE FUNCTION TSystem.LogAlways(source VARCHAR, message TEXT, VARIADIC arr_msg_txt VARCHAR[]) RETURNS VOID AS $$
      BEGIN
        PERFORM TSystem.LogMessage(6, source, TSystem.LogFormat(message, False, VARIADIC arr_msg_txt), 0, NULL);
        RETURN;
      END $$ LANGUAGE plpgsql;
    --
--

-- LogFormat
-- Eingangsparameter:
    -- msg TEXT - Message-Text
    -- linebreaks BOOLEAN - Message-Info bilden mit Zeilenumbruche
    -- VARIADIC arr_msg_txt VARCHAR[] - Message-Info als Array
CREATE OR REPLACE FUNCTION TSystem.LogFormat(IN msg TEXT, IN linebreaks BOOLEAN, VARIADIC arr_msg_txt VARCHAR[] DEFAULT NULL) RETURNS TEXT AS $$
  DECLARE msg_txt TEXT;
  BEGIN
    msg_txt:= format('%s', arr_msg_txt);    -- Text zusammenbauen
    -- ungewünschte Zeichen abschneiden
    msg_txt:= TRIM(leading '{' FROM msg_txt);
    msg_txt:= TRIM(trailing '}' FROM msg_txt);
    msg_txt:= REPLACE(msg_txt, '= ","', '= ');
    msg_txt:= REPLACE(msg_txt, '=","', '=');
    msg_txt:= REPLACE(msg_txt, '",', '');
    IF linebreaks THEN               -- Zeilenunbruche erstellen
        msg_txt:= REPLACE(msg_txt, '","', E'\n');
        msg_txt:= REPLACE(msg_txt, ',"' , E'\n');
    END IF;
    msg_txt:= REPLACE(msg_txt, '"', '');
    IF linebreaks THEN
        msg_txt:= msg || E'\n' || msg_txt;
    ELSE
        msg_txt:= msg || '   ' || msg_txt;
    END IF;

    RETURN msg_txt;
  END $$ LANGUAGE plpgsql;
--

-- LogText-Formatierung, wenn Record oder Teil von Record gespeichert soll, Record ist als JSON Eingangsparameter
CREATE OR REPLACE FUNCTION TSystem.LogFormat(IN msg TEXT, IN linebreaks BOOLEAN, IN js JSON) RETURNS TEXT AS $$
  DECLARE msg_txt TEXT:= '';
          rec RECORD;
  BEGIN
    FOR rec IN SELECT * FROM json_each(js) LOOP
        IF NOT linebreaks THEN
            -- msg_txt:= msg_txt || '"' || rec.key::VARCHAR || '" = ' || rec.value::VARCHAR || ',';  -- TODO JSON-Variante
            msg_txt:= msg_txt || rec.key::VARCHAR || ' = ' || rec.value::VARCHAR || ', ';
        ELSE
            msg_txt:= msg_txt || rec.key::VARCHAR || ' = ' || rec.value::VARCHAR;
            msg_txt:= msg_txt ||  E'\n';
        END IF;
    END LOOP;

    msg_txt := TRIM(msg_txt);

    IF NOT linebreaks THEN
        -- msg_txt:= msg || '   {' || SUBSTRING(msg_txt from 1 for char_length(msg_txt) - 1) || '}';  -- TODO JSON-Variante
        msg_txt:= msg || '   ' || SUBSTRING(msg_txt from 1 for char_length(msg_txt) - 1);
    ELSE
        msg_txt:= msg || E'\n' || msg_txt;
    END IF;

    RETURN msg_txt;
  END $$ LANGUAGE plpgsql;
--

-- DB-Logfunktionen als Eingabe u. Debughilfe    ******* Ende *******

CREATE OR REPLACE FUNCTION tsystem.selects_count__control(
    IN _tabname      varchar,
    IN _select_q     text,
    IN _select_z     text
  ) RETURNS void AS $$
  DECLARE _count_q   integer;   --- Quelle
          _count_z   integer;   --- Ziel
  BEGIN

    EXECUTE _select_q INTO _count_q;
    EXECUTE _select_z INTO _count_z;

    RAISE NOTICE '_select_z = %', _select_z;
    RAISE NOTICE '_count1 = %, _count2 = %', _count_q, _count_z;

    IF  _count_q <> _count_z THEN
        PERFORM PRODAT_MESSAGE( format( lang_text( 29789 ), _tabname, _count_q, _count_z ), 'Information');
    END IF;

 END $$ LANGUAGE plpgsql;
---



